(function($) {
    'use strict';
    
    // LearnDash Smart Engine Admin Interface
    const LDSmartAdmin = {
        
        init: function() {
            this.bindEvents();
            this.initToggle();
            this.initNotifications();
            this.hideWordPressNotices();
            console.log('LearnDash Smart Engine Admin loaded');
        },
        
        bindEvents: function() {
            // Plugin toggle handler
            $(document).on('change', '#ld-smart-plugin-toggle', this.handlePluginToggle.bind(this));
            
            // Tab switching (if needed for AJAX)
            $(document).on('click', '.ld-smart-tab', this.handleTabSwitch.bind(this));
            
            // Form submission enhancements
            $(document).on('submit', '.ld-smart-content form', this.handleFormSubmit.bind(this));
        },
        
        initToggle: function() {
            const toggle = $('#ld-smart-plugin-toggle');
            if (toggle.length) {
                this.updateToggleStatus(toggle.is(':checked'));
            }
        },
        
        initNotifications: function() {
            // Auto-hide notifications after 5 seconds
            setTimeout(function() {
                $('.ld-smart-notification').fadeOut(300);
            }, 5000);
        },
        
        hideWordPressNotices: function() {
            // Hide WordPress admin notices on our plugin page
            if (window.location.href.indexOf('ld-smart-settings') !== -1) {
                // Hide immediately
                $('.notice, .error, .updated, .update-nag').hide();
                
                // Hide notices that might be added later via AJAX
                const observer = new MutationObserver(function(mutations) {
                    mutations.forEach(function(mutation) {
                        mutation.addedNodes.forEach(function(node) {
                            if (node.nodeType === 1) { // Element node
                                const $node = $(node);
                                if ($node.hasClass('notice') || $node.hasClass('error') || 
                                    $node.hasClass('updated') || $node.hasClass('update-nag') ||
                                    $node.find('.notice, .error, .updated, .update-nag').length) {
                                    $node.hide();
                                    $node.find('.notice, .error, .updated, .update-nag').hide();
                                }
                            }
                        });
                    });
                });
                
                // Start observing
                observer.observe(document.body, {
                    childList: true,
                    subtree: true
                });
                
                // Also hide after page load
                setTimeout(() => {
                    $('.notice, .error, .updated, .update-nag').hide();
                }, 100);
            }
        },
        
        handlePluginToggle: function(e) {
            const $toggle = $(e.target);
            const isEnabled = $toggle.is(':checked');
            const originalState = $toggle.data('original-state');
            
            // Update visual status immediately
            this.updateToggleStatus(isEnabled);
            
            // If state changed, save via AJAX
            if ((isEnabled ? '1' : '0') !== originalState) {
                this.savePluginState(isEnabled, $toggle);
            }
        },
        
        updateToggleStatus: function(isEnabled) {
            const statusText = $('#ld-smart-toggle-status');
            if (isEnabled) {
                statusText.text(statusText.data('active-text') || 'Active')
                          .removeClass('inactive')
                          .addClass('active');
            } else {
                statusText.text(statusText.data('inactive-text') || 'Inactive')
                          .removeClass('active')
                          .addClass('inactive');
            }
        },
        
        savePluginState: function(isEnabled, $toggle) {
            const data = {
                action: 'ld_smart_toggle_plugin',
                enabled: isEnabled ? '1' : '0',
                nonce: ldSmartAdmin.nonce || ''
            };
            
            // Show loading state
            this.showNotification('Saving...', 'info', false);
            
            $.ajax({
                url: ajaxurl,
                type: 'POST',
                data: data,
                success: (response) => {
                    if (response.success) {
                        $toggle.data('original-state', isEnabled ? '1' : '0');
                        this.showNotification('Settings saved successfully!', 'success');
                    } else {
                        // Revert toggle on error
                        $toggle.prop('checked', !isEnabled);
                        this.updateToggleStatus(!isEnabled);
                        this.showNotification('Error saving settings: ' + (response.data || 'Unknown error'), 'error');
                    }
                },
                error: () => {
                    // Revert toggle on error
                    $toggle.prop('checked', !isEnabled);
                    this.updateToggleStatus(!isEnabled);
                    this.showNotification('Network error occurred while saving settings.', 'error');
                }
            });
        },
        
        handleTabSwitch: function(e) {
            // Add any tab-specific logic here if needed
            const $tab = $(e.currentTarget);
            const tabId = $tab.data('tab');
            
            // Could be used for AJAX tab loading in the future
            console.log('Switching to tab:', tabId);
        },
        
        handleFormSubmit: function(e) {
            const $form = $(e.target);
            const $submitBtn = $form.find('input[type="submit"]');
            
            // Add loading state to submit button
            $submitBtn.prop('disabled', true)
                     .val('Saving...')
                     .addClass('ld-smart-loading');
            
            // Re-enable after a delay (WordPress will redirect on success)
            setTimeout(function() {
                $submitBtn.prop('disabled', false)
                         .val('Save Changes')
                         .removeClass('ld-smart-loading');
            }, 3000);
        },
        
        showNotification: function(message, type = 'info', autoHide = true) {
            const notificationHtml = `
                <div class="ld-smart-notification ${type}">
                    <span class="dashicons dashicons-${this.getNotificationIcon(type)}"></span>
                    <span>${message}</span>
                </div>
            `;
            
            const $container = $('#ld-smart-notifications');
            $container.html(notificationHtml);
            
            if (autoHide) {
                setTimeout(function() {
                    $container.find('.ld-smart-notification').fadeOut(300, function() {
                        $(this).remove();
                    });
                }, 3000);
            }
        },
        
        getNotificationIcon: function(type) {
            const icons = {
                'success': 'yes-alt',
                'error': 'dismiss',
                'warning': 'warning',
                'info': 'info'
            };
            return icons[type] || 'info';
        }
    };
    
    // Initialize when document is ready
    $(document).ready(function() {
        LDSmartAdmin.init();
    });
    
    // Expose to global scope if needed
    window.LDSmartAdmin = LDSmartAdmin;
    
})(jQuery); 